/*! \file tlk_cert_api_ieee_1667.h 
**********************************************************************************
*Title:                         Discretix Certificate Handling Toolkit source file
*
* Filename:                     tlk_cert_api_ieee_1667.h 
*
* Project, Target, subsystem:   Toolkit, Certificate Handling, IEEE 1667 Adaptation
* 
* Created:                      11.02.2009
*
* Modified:                     11.02.2009
*
* \Author                       Ira Boguslavsky
*
* \Remarks
*           Copyright (C) 2007 by Discretix Technologies Ltd. All Rights reserved.
**********************************************************************************/
#ifndef _TLK_CERT_API_IEEE_1667_H_
#define _TLK_CERT_API_IEEE_1667_H_

#include "tlk_cert_types.h"
#include "tlk_cert_types_ieee_1667.h"
#ifdef __cplusplus
extern "C"
{
#endif


/**
* @brief    This function initializes context object. 
*           The context object will be set at a starting position and it will be marked as just initialized.
* @note     User is responsible to provide of this context object.
*
* @param contextObject_ptr      [out]        - Pointer to the calculated certificate identifier
*
* @return TLK_CERT_RC_OK                                            - The operation completed successfully.
* @return TLK_CERT_RC_ERROR_API_NULL_POINTER                        - Null pointer received as input
* @return TLK_CERT_RC_ERROR_API_CONTEXT_OBJ_MAC_CALCULATE_FAILED    - Context Object MAC calculation failed because of CRYS_HASH problems
* @return TLK_CERT_RC_ERROR_X509_UNSUPPORTED_VERSION                - X.509 Parser got unsupported certificate version
* @return TLK_CERT_RC_ERROR_X509_TOO_MANY_EXTENDED_KEY_USAGES       - X.509 Parser got too many extended key usages
* @return TLK_CERT_RC_ERROR_X509_UNRECOGNIZED_CRITICAL_EXTENSION    - X.509 Parser got unrecognized critical extension
* @return TLK_CERT_RC_ERROR_X509_INVALID_TIME                       - X.509 Parser got invalid time structure (buffer, the buffer length incorrect, etc)   
* @return TLK_CERT_RC_ERROR_ASN1_BAD_ARGUMENTS                      - ASN1 Parser got invalid arguments
* @return TLK_CERT_RC_ERROR_ASN1_ITEM_NOT_FOUND                     - ASN1 Parser supposed to get specific structure of the item
* @return TLK_CERT_RC_ERROR_ASN1_OVERFLOW                           - ASN1 Parser got invalid tag of one of the fields in the structure
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_CLASS              - ASN1 Parser got unexpected item class
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_ENCODING           - ASN1 Parser got unexpected item encoding
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_TAG                - ASN1 Parser got unexpected item tag
* @return TLK_CERT_RC_ERROR_ASN1_INVALID_SEQ_STRUCTURE              - ASN1 Parser got invalid sequence structure
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_OID                     - ASN1 Parser got unexpected object identifier
* @return TLK_CERT_RC_ERROR_BUFF_SEEK_FAILED                        - Buffer seek operation failed
* @return TLK_CERT_RC_ERROR_BUFF_BUFFER_IS_NOT_BIG_ENOUGH           - Buffer is not big enough
* @return TLK_CERT_RC_ERROR_BUFF_ITEM_NOT_FOUND                     - Searching in buffer failed
* @return TLK_CERT_RC_ERROR_BUFF_NULL_POINTER                       - Null pointer received as input
* @return TLK_CERT_RC_ERROR_BUFF_NO_MORE_PLACE                      - The buffer capacity is over
*/
DxError_t TLK_CERT_1667_ChainValidationInit(           TLK_CERT_1667_ContextObj_t   *contextObject_ptr);

/**
* @brief    This a main function which covers all three steps of the chain validation algorithm:
*           Initialization, Processing and Termination. It will go through every 
*           field in the certificate structure and will perform validation according to the step and data 
*           which is stored in the context object. The user will be responsible to provide the context object, 
*           which will be an input for the every step. 
*           Also this object will be protected cryptographically (MAC) or will be allocated in the 
*           trust environment. 
* @note     User is responsible for contextObject_ptr (It is MAC protected)
*
* @param prevCertificate_ptr       [in]      - Pointer to the previous certificate.
* @param currCertificate_ptr       [in]      - Pointer to the certificate to be processed.
* @param contextObject_ptr         [in/out]  - Pointer to the context object
* @param authAction                [in]      - One of the following:
*                                              Begin    - Validate the PCp certificate
*                                              Continue - Validate against previous certificate
*                                              Final    - Validate against previous certificate and verify challenge based on extracted Public Key
*                                              Self     - Validate and verify challenge based on PCp Public Key
* @param workspace_ptr             [in]      - Pointer to the workspace. Note: The user is responsible for workspace allocation.
* @param workspaceSizeInBytes      [in]      - Length of the workspace in bytes
*
* @return TLK_CERT_RC_OK                                            - The operation completed successfully.
* @return TLK_CERT_RC_ERROR_API_NULL_POINTER                        - Null pointer received as input
* @return TLK_CERT_RC_ERROR_API_CONTEXT_OBJ_MAC_CALCULATE_FAILED    - Context Object MAC calculation failed because of CRYS_HASH problems
* @return TLK_CERT_RC_ERROR_INTEGRITY                               - Context object integrity compromised
* @return TLK_CERT_RC_ERROR_X509_UTIL_NOT_BEFORE_TIME_FIELD_FAILED  - Problem to convert validNotBefore field from date format to seconds_sinse format
* @return TLK_CERT_RC_ERROR_X509_UTIL_NOT_AFTER_TIME_FIELD_FAILED   - Problem to convert validNotAfter field from date format to seconds_sinse format
* @return TLK_CERT_RC_ERROR_X509_UTIL_VALIDITY_PERIOD_VERIFY_FAILED - The current certificate is expired
* @return TLK_CERT_RC_ERROR_X509_UNSUPPORTED_VERSION                - X.509 Parser got unsupported certificate version
* @return TLK_CERT_RC_ERROR_X509_TOO_MANY_EXTENDED_KEY_USAGES       - X.509 Parser got too many extended key usages
* @return TLK_CERT_RC_ERROR_X509_UNRECOGNIZED_CRITICAL_EXTENSION    - X.509 Parser got unrecognized critical extension
* @return TLK_CERT_RC_ERROR_X509_INVALID_TIME                       - X.509 Parser got invalid time structure (buffer, the buffer length incorrect, etc)   
* @return TLK_CERT_RC_ERROR_X509_UTIL_SUBJECT_NAME_INVALID          - Subject name of the current ("parent") certificate is differ from the issuer name of the "child" certificate
* @return TLK_CERT_RC_ERROR_X509_UTIL_EE_SUBJECT_NAME_MISSING       - Subject name in end-entity certificate missing
* @return TLK_CERT_RC_ERROR_X509_UTIL_TR_SUBJECT_NAME_MISSING       - Subject name in trusted root certificate missing
* @return TLK_CERT_RC_ERROR_X509_UTIL_SUBJECT_NAME_MISSING          - Subject name in certificate missing
* @return TLK_CERT_RC_ERROR_X509_UTIL_USSUER_NAME_MISSING           - Issuer name in certificate missing
* @return TLK_CERT_RC_ERROR_ASN1_BAD_ARGUMENTS                      - ASN1 Parser got invalid arguments
* @return TLK_CERT_RC_ERROR_ASN1_ITEM_NOT_FOUND                     - ASN1 Parser supposed to get specific structure of the item
* @return TLK_CERT_RC_ERROR_ASN1_OVERFLOW                           - ASN1 Parser got invalid tag of one of the fields in the structure
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_CLASS              - ASN1 Parser got unexpected item class
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_ENCODING           - ASN1 Parser got unexpected item encoding
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_TAG                - ASN1 Parser got unexpected item tag
* @return TLK_CERT_RC_ERROR_ASN1_INVALID_SEQ_STRUCTURE              - ASN1 Parser got invalid sequence structure
* @return TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_OID                     - ASN1 Parser got unexpected object identifier
* @return TLK_CERT_RC_ERROR_BUFF_SEEK_FAILED                        - Buffer seek operation failed
* @return TLK_CERT_RC_ERROR_BUFF_BUFFER_IS_NOT_BIG_ENOUGH           - Buffer is not big enough
* @return TLK_CERT_RC_ERROR_BUFF_ITEM_NOT_FOUND                     - Searching in buffer failed
* @return TLK_CERT_RC_ERROR_BUFF_NULL_POINTER                       - Null pointer received as input
* @return TLK_CERT_RC_ERROR_BUFF_NO_MORE_PLACE                      - The buffer capacity is over
*/
DxError_t TLK_CERT_1667_ChainValidationProcess( const  TLK_CERT_Buffer_t            *prevCertificate_ptr,
                                                const  TLK_CERT_Buffer_t            *currCertificate_ptr, 
                                                       TLK_CERT_1667_ContextObj_t   *contextObject_ptr,
                                                       TLK_CERT_1667_AuthAction_t    authAction,
                                                       DxUint8_t                    *workspace_ptr,
                                                const  DxUint32_t                    workspaceSizeInBytes);
 
/**
* @brief    The CVPO will be filled from the context object. 
* @note     User is responsible for contextObject_ptr (It is MAC protected)
*
* @param cvpoObject_ptr         [out]        - Pointer to the certificate
* @param contextObject_ptr      [in]         - Pointer to the calculated certificate identifier
*
* @return TLK_CERT_RC_OK                                            - The operation completed successfully.
* @return TLK_CERT_RC_ERROR_API_NULL_POINTER                        - Null pointer received as input
*/
DxError_t TLK_CERT_1667_ChainValidationTerminate(const  TLK_CERT_1667_ContextObj_t   *contextObject_ptr,
                                                        TLK_CERT_1667_CPVO_t         *cvpoObject_ptr);

#ifdef __cplusplus
}
#endif /*extern "C"*/
#endif

